%{
Copyright (c) 2011, MIT.
Developed with the sponsorship of the Defense Advanced Research Projects
Agency (DARPA).

Permission is hereby granted, free of charge, to any person obtaining a copy
of this data, including any software or models in source or binary form, as
well as any drawings, specifications, and documentation (collectively "the
Data"), to deal in the Data without restriction, including without
limitation the rights to use, copy, modify, merge, publish, distribute,
sublicense, and/or sell copies of the Data, and to permit persons to whom
the Data is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE DATA OR
THE USE OR OTHER DEALINGS IN THE DATA.
%}

% Matt Fitzgerald
% cost and attribute calculator

function [cost attributes intermediate] = designCalculator(designVariables,DFClevel,tech)

numDesigns = size(designVariables,1);
cost = zeros(numDesigns,1);
attributes = zeros(numDesigns,3);
intermediate = zeros(numDesigns,5);

% loops through included design variable options
for d = 1:numDesigns
    if strcmp('Low',designVariables{d,1})
        payload = 300;
    elseif strcmp('Medium',designVariables{d,1})
        payload = 1000;
    elseif strcmp('High',designVariables{d,1})
        payload = 3000;
    elseif strcmp('Extreme',designVariables{d,1})
        payload = 5000;
    end
    
    if strcmp('Storable bi',designVariables{d,2})
        Isp = 300;
        baseMass = 0;
        massFrac = .12;
        fast = 1;
    elseif strcmp('Cryo',designVariables{d,2})
        if strcmp(tech,'future')
            Isp = 550;
        else
            Isp = 450;
        end
        baseMass = 0;
        massFrac = .13;
        fast = 1;
    elseif strcmp('Electric',designVariables{d,2})
        Isp = 3000;
        baseMass = 25;
        if strcmp(tech,'future')
            massFrac = .25;
        else
            massFrac = .3;
        end
        fast = 0;
    elseif strcmp('Nuclear',designVariables{d,2})
        Isp = 1500;
        if strcmp(tech,'future')
            baseMass = 600;
        else
            baseMass = 1000;
        end
        massFrac = .2;
        fast = 1;
    end
    
    propellant = designVariables{d,3};
    
    dryMass = 2*payload + baseMass + massFrac*propellant;
    % add in DFC cost penalty
    if DFClevel == 1
        dryMass = dryMass * 1.1;
    elseif DFClevel == 2
        dryMass = dryMass * 1.2;
    end
    
    wetMass = dryMass + propellant;
    
    if strcmp(tech,'future')
        cost(d,1) = (15*wetMass + 150*dryMass)/1000;
    else
        cost(d,1) = (10*wetMass + 150*dryMass)/1000;
    end
    % cost is in $millions

    
    % find deltaV
    gravity = 9.8;
    deltaV = gravity*Isp*log(wetMass/dryMass);
    % set attributes
    attributes(d,1) = payload;
    attributes(d,2) = fast;
    attributes(d,3) = deltaV;
    % save intermediates as well
    intermediate(d,1) = Isp;
    intermediate(d,2) = massFrac;
    intermediate(d,3) = baseMass;
    intermediate(d,4) = dryMass;
    intermediate(d,5) = wetMass;
end


end